;; dx4.ru-cutting.lsp - LISP-,            CAD
;; : NanoCAD, AutoCAD, GstarCAD
;; : 2.3.0


(setq *selected-objects* nil)
(setq *manual-parts* nil)
(setq *material-width* 2800)
(setq *material-height* 2070)
(setq *cutting-margin* 10)
(setq *blade-thickness* 3)
(setq *algorithm* "0")
(setq *allow-rotation* "1")
(setq *show-waste* "0")
(setq *one-color* "0")
(setq *show-metrics* "1")
(setq *api-key* "demoplugin")
(setq *api-url* "https://dx4.ru/cut/api-v2/")
(setq *previous-results* nil)
(setq *plugin-session-id* nil)
(setq *cad-platform* nil)


(defun get-cad-platform ( / prod program)

  (setq prod (getvar "PRODUCT"))
  (if (null prod) (setq prod ""))     
  (setq prod (strcase prod))          


  (setq program (getvar "PROGRAM"))
  (if (null program) (setq program ""))
  (setq program (strcase program))


  (princ (strcat "\n[DEBUG] Platform detection: PRODUCT='" prod "' PROGRAM='" program "'"))

  (cond

    ((or (wcmatch prod "*NANO*")
         (wcmatch program "*NANO*"))
     "nanocad")
    

    ((or (wcmatch prod "*GSTAR*")
         (wcmatch program "*GCAD*")    
         (wcmatch program "*GSTAR*"))
     "gstarcad")
    

    ((wcmatch prod "*BRICS*")
     "bricscad")
    

    ((wcmatch prod "*ZW*")
     "zwcad")
    

    ((or (= prod "AUTOCAD")
         (wcmatch prod "AUTOCAD*")    
         (wcmatch program "*ACAD*"))  
     "autocad")
    

    (T "unknown")
  )
)



(defun get-os-type ()
  (cond

    ((or (getenv "OS")
         (getenv "WINDIR")
         (findfile "C:\\Windows\\"))
     "windows")
    

    ((or (getenv "HOME")
         (getenv "XDG_CONFIG_HOME")
         (= (type (getenv "WAYLAND_DISPLAY")) 'STR)
         (= (type (getenv "DISPLAY")) 'STR))
     "linux")
    

    ((or (getenv "DYLD_LIBRARY_PATH")
         (findfile "/Applications/"))
     "macos")
    
    (T "unknown")
  )
)


(defun get-session-id ()
  (if (null *plugin-session-id*)
    (progn
      (setq *plugin-session-id* 
        (strcat 
          (rtos (getvar "CDATE") 2 8) 
          "-"
          (rtos (rem (getvar "MILLISECS") 10000) 2 0)
        )
      )
      (save-settings)
    )
  )
  *plugin-session-id*
)


(defun get-timestamp ( / old-dimzin res)

  (setq old-dimzin (getvar "DIMZIN"))

  (setvar "DIMZIN" 0)
  

  (setq res (rtos (getvar "CDATE") 2 8))
  

  (setvar "DIMZIN" old-dimzin)
  
  res
)


(defun get-dwg-hash ()
  (itoa (rem (strlen (getvar "DWGNAME")) 65536))
)


(defun get-cad-version ()
  (if (getvar "ACADVER")
    (getvar "ACADVER")
    "unknown"
  )
)


(defun get-plugin-version ()
  "2.3.0"
)




(defun get-settings-path ()
  (strcat (getenv "USERPROFILE") "\\Documents\\dx4.ru_cutting_settings.txt")
)


(defun load-settings ()
  (setq settings-file (get-settings-path))
  (princ (strcat "\n   : " settings-file))
  
  (setq *cad-platform* (get-cad-platform))
  (princ (strcat "\n CAD-: " *cad-platform*))
  (princ (strcat "\n : " (get-os-type)))
  
  (if (findfile settings-file)
    (progn
      (setq f (open settings-file "r"))
      (if f
        (progn
          (setq line-count 0)
          (setq temp-list '())
          
          (while (setq line (read-line f))
            (setq temp-list (cons line temp-list))
            (setq line-count (1+ line-count))
          )
          (close f)
          
          (setq temp-list (reverse temp-list))
          (princ (strcat "\n   : " (itoa line-count)))
          
          (cond
            ((>= line-count 11)
              (setq *api-key* (nth 0 temp-list))
              (setq *material-width* (atof (nth 1 temp-list)))
              (setq *material-height* (atof (nth 2 temp-list)))
              (setq *cutting-margin* (atof (nth 3 temp-list)))
              (setq *blade-thickness* (atof (nth 4 temp-list)))
              (setq *algorithm* (nth 5 temp-list))
              (setq *allow-rotation* (nth 6 temp-list))
              (setq *show-waste* (nth 7 temp-list))
              (setq *one-color* (nth 8 temp-list))
              (setq *plugin-session-id* (nth 9 temp-list))
              (setq *show-metrics* (nth 10 temp-list))
              (princ "\n?   (   show-metrics)")
            )
            ((>= line-count 10)
              (setq *api-key* (nth 0 temp-list))
              (setq *material-width* (atof (nth 1 temp-list)))
              (setq *material-height* (atof (nth 2 temp-list)))
              (setq *cutting-margin* (atof (nth 3 temp-list)))
              (setq *blade-thickness* (atof (nth 4 temp-list)))
              (setq *algorithm* (nth 5 temp-list))
              (setq *allow-rotation* (nth 6 temp-list))
              (setq *show-waste* (nth 7 temp-list))
              (setq *one-color* (nth 8 temp-list))
              (setq *plugin-session-id* (nth 9 temp-list))
              (setq *show-metrics* "1")
              (princ "\n?   ( , show-metrics   )")
            )
            ((>= line-count 9)
              (setq *api-key* (nth 0 temp-list))
              (setq *material-width* (atof (nth 1 temp-list)))
              (setq *material-height* (atof (nth 2 temp-list)))
              (setq *cutting-margin* (atof (nth 3 temp-list)))
              (setq *blade-thickness* (atof (nth 4 temp-list)))
              (setq *algorithm* (nth 5 temp-list))
              (setq *allow-rotation* (nth 6 temp-list))
              (setq *show-waste* (nth 7 temp-list))
              (setq *one-color* (nth 8 temp-list))
              (setq *plugin-session-id* (get-session-id))
              (setq *show-metrics* "1")
              (princ "\n?   ( , session-id  show-metrics )")
            )
            ((>= line-count 1)
              (setq *api-key* (nth 0 temp-list))
              (setq *plugin-session-id* (get-session-id))
              (setq *show-metrics* "1")
              (princ "\n?   API  (   )")
            )
            (t
              (princ "\n?   ")
              (setq *plugin-session-id* (get-session-id))
              (setq *show-metrics* "1")
            )
          )
        )
        (princ "\n?       ")
      )
    )
    (progn
      (princ "\n?    ,     ")
      (setq *plugin-session-id* (get-session-id))
      (setq *show-metrics* "1")
      (save-settings)
    )
  )
)


(defun save-settings ()
  (setq settings-file (get-settings-path))
  (princ (strcat "\n  : " settings-file))
  
  (if (null *plugin-session-id*)
    (setq *plugin-session-id* (get-session-id))
  )
  
  (setq f (open settings-file "w"))
  (if f
    (progn
      (write-line *api-key* f)
      (write-line (rtos *material-width* 2 0) f)
      (write-line (rtos *material-height* 2 0) f)
      (write-line (rtos *cutting-margin* 2 0) f)
      (write-line (rtos *blade-thickness* 2 0) f)
      (write-line *algorithm* f)
      (write-line *allow-rotation* f)
      (write-line *show-waste* f)
      (write-line *one-color* f)
      (write-line *plugin-session-id* f)
      (write-line *show-metrics* f)
      (close f)
      (princ "\n?    (11 )")
      T
    )
    (progn
      (princ "\n? :       ")
      nil
    )
  )
)




(defun c:CUTTING (/ dcl_id result)
  (load-settings)
  
  (princ (strcat "\n\n===    ==="))
  (princ (strcat "\nCAD-: " *cad-platform*))
  (princ (strcat "\n CAD: " (get-cad-version)))
  (princ (strcat "\n: " (get-os-type)))
  (princ (strcat "\n : " (get-plugin-version)))
  (princ (strcat "\nSession ID: " (get-session-id)))
  (princ (strcat "\n: " (getvar "DWGNAME")))
  (princ "\n==============================\n")
  
  (setq dcl_id (load_dialog "dx4.ru-cutting.dcl"))
  (if (< dcl_id 0)
    (progn
      (alert "    dx4.ru-cutting.dcl!")
      (exit)
    )
  )
  
  (if (not (new_dialog "cutting_dialog" dcl_id))
    (progn
      (alert "   !")
      (unload_dialog dcl_id)
      (exit)
    )
  )
  
  (init_dialog)
  
  (action_tile "select_objects" "(done_dialog 2)")
  (action_tile "manual_input" "(done_dialog 4)")
  (action_tile "calculate" "(done_dialog 3)")
  (action_tile "accept" "(done_dialog 1)")
  (action_tile "cancel" "(done_dialog 0)")
  
  (action_tile "api_key" "(setq *api-key* $value)")
  (action_tile "material_width" "(setq *material-width* (atof $value))")
  (action_tile "material_height" "(setq *material-height* (atof $value))")
  (action_tile "cutting_margin" "(setq *cutting-margin* (atof $value))")
  (action_tile "blade_thickness" "(setq *blade-thickness* (atof $value))")
  (action_tile "algorithm" "(setq *algorithm* $value)")
  (action_tile "allow_rotation" "(setq *allow-rotation* $value)")
  (action_tile "show_waste" "(setq *show-waste* $value)")
  (action_tile "one_color" "(setq *one-color* $value)")
  (action_tile "show_metrics" "(setq *show-metrics* $value)")
  
  (setq result (start_dialog))
  (unload_dialog dcl_id)
  
  (cond
    ((= result 0)
      (princ "\n .")
    )
    ((= result 1)
      (if (save-settings)
        (princ "\n?  ,  .")
        (princ "\n?    ,  .")
      )
    )
    ((= result 2)
      (select_objects_action)
      (c:CUTTING)
    )
    ((= result 3)
      (if (save-settings)
        (princ "\n?    .")
        (princ "\n?      .")
      )
      (calculate_nesting_action)
    )
    ((= result 4)
      (manual_input_action)
      (c:CUTTING)
    )
  )
  
  (princ)
)


(defun init_dialog ()
  (set_tile "api_key" *api-key*)
  (set_tile "material_width" (rtos *material-width* 2 0))
  (set_tile "material_height" (rtos *material-height* 2 0))
  (set_tile "cutting_margin" (rtos *cutting-margin* 2 0))
  (set_tile "blade_thickness" (rtos *blade-thickness* 2 0))
  (set_tile "algorithm" *algorithm*)
  (set_tile "allow_rotation" *allow-rotation*)
  (set_tile "show_waste" *show-waste*)
  (set_tile "one_color" *one-color*)
  (set_tile "show_metrics" *show-metrics*)
  
  (if *manual-parts*
    (set_tile "objects_count" (strcat " : " (itoa (length *manual-parts*)) " "))
    (set_tile "objects_count" (strcat " : " (itoa (length *selected-objects*))))
  )
  
  (set_tile "status" "  ")
)


(defun select_objects_action ()
  (princ "\n  ...")
  
  (setq ss (ssget '((0 . "LWPOLYLINE"))))
  
  (if ss
    (progn
      (setq *selected-objects* (ssget-to-list ss))
      (setq *manual-parts* nil)
      (princ (strcat "\n : " (itoa (length *selected-objects*))))
      (princ "\n  .")
    )
    (progn
      (setq *selected-objects* nil)
      (princ "\n  ")
    )
  )
)


(defun manual_input_action (/ dcl_id result input_text parts_list)
  (setq dcl_id (load_dialog "dx4.ru-cutting.dcl"))
  
  (if (not (new_dialog "manual_input_dialog" dcl_id))
    (progn
      (alert "    !")
      (unload_dialog dcl_id)
      (exit)
    )
  )
  

  (set_tile "info_label" ": , x, *")
  (set_tile "label" ": , x, *")
  

  (if (and *manual-parts* (> (length *manual-parts*) 0))
    (set_tile "parts_input" (manual-parts-to-string *manual-parts*))
    (set_tile "parts_input" "200x100-2,300*200-1,150x150-3")
  )
  

  (action_tile "accept" "(save_input) (done_dialog 1)")
  (action_tile "cancel" "(done_dialog 0)")
  

  (defun save_input ()
    (setq input_text (get_tile "parts_input"))
    (princ (strcat "\n : " (if input_text input_text "")))
  )
  
  (setq result (start_dialog))
  (unload_dialog dcl_id)
  

  (if (= result 1)
    (progn

      (if (null input_text)
        (progn
          (princ "\ninput_text is nil, trying to get value again...")
          (setq dcl_id (load_dialog "dx4.ru-cutting.dcl"))
          (if (new_dialog "manual_input_dialog" dcl_id)
            (progn
              (setq input_text (get_tile "parts_input"))
              (unload_dialog dcl_id)
            )
          )
        )
      )
      

      (if (null input_text)
        (setq input_text "")
      )
      
      (princ (strcat "\n : '" input_text "'"))
      

      (setq parts_list (simple-parse-input input_text))
      
      (if parts_list
        (progn
          (setq *manual-parts* parts_list)
          (setq *selected-objects* nil)
          (princ (strcat "\n: " (itoa (length parts_list)) " "))
          (alert (strcat "  " (itoa (length parts_list)) " "))
        )
        (progn
          (alert " ! : x-\n : x  *\n: 200x100-2,300*200-1")
          (manual_input_action) 
        )
      )
    )
    (princ "\n  ")
  )
)


(defun simple-parse-input (input-text / items parts-list item parts)
  (princ (strcat "\n: '" input-text "'"))
  
  (if (or (null input-text) (= input-text ""))
    (progn
      (princ "\n ")
      (return nil)
    )
  )
  
  (setq items (split-string-simple input-text ","))
  (setq parts-list '())
  
  (princ (strcat "\n : " (itoa (length items))))
  
  (foreach item items
    (setq item (vl-string-trim " \t\r\n" item))
    (princ (strcat "\n: '" item "'"))
    
    (if (and (> (strlen item) 0) (setq parts (parse-single-part item)))
      (progn
        (setq parts-list (cons parts parts-list))
        (princ (strcat " -> OK: " (itoa (car parts)) "x" (itoa (cadr parts)) "-" (itoa (caddr parts))))
      )
      (if (> (strlen item) 0)
        (princ " -> ERROR")
      )
    )
  )
  
  (if (> (length parts-list) 0)
    (progn
      (princ (strcat "\n : " (itoa (length parts-list)) " "))
      (reverse parts-list)
    )
    (progn
      (princ "\n     ")
      nil
    )
  )
)


(defun parse-single-part (item / x-pos dash-pos width height quantity)
  (princ (strcat "\n   : '" item "'"))
  

  (setq x-pos (vl-string-position (ascii "x") item))

  (if (not x-pos)
    (setq x-pos (vl-string-position (ascii "*") item))
  )
  
  (setq dash-pos (vl-string-position (ascii "-") item))
  
  (if (and x-pos dash-pos (> x-pos 0) (> dash-pos x-pos))
    (progn
      (setq width (atoi (substr item 1 x-pos)))
      (setq height (atoi (substr item (+ x-pos 2) (- dash-pos x-pos 1))))
      (setq quantity (atoi (substr item (+ dash-pos 2))))
      
      (princ (strcat "\n    width=" (itoa width) " height=" (itoa height) " quantity=" (itoa quantity)))
      
      (if (and (> width 0) (> height 0) (> quantity 0))
        (list width height quantity)
        (progn
          (princ "\n       ")
          nil
        )
      )
    )
    (progn
      (princ "\n       x-  *-")
      nil
    )
  )
)


(defun split-string-simple (str delimiter / pos result)
  (setq result '())
  
  (while (setq pos (vl-string-search delimiter str))
    (setq result (cons (substr str 1 pos) result))
    (setq str (substr str (+ pos 2)))
  )
  
  (reverse (cons str result))
)


(defun universal-parse-input (input-text / lines parts-list valid-parts-count)
  (princ "\n   ...")
  

  (setq input-text (vl-string-translate ",;" "||" input-text))  
  (setq input-text (vl-string-translate "\r\n" "||" input-text)) 
  
  (princ (strcat "\n  : " input-text))
  
  (setq lines (split-string input-text "||"))
  (setq parts-list '())
  (setq valid-parts-count 0)
  
  (princ (strcat "\n  : " (itoa (length lines))))
  
  (foreach line lines
    (setq line (vl-string-trim " \t\r\n,;" line))
    
    (if (> (strlen line) 0)
      (progn
        (princ (strcat "\n: '" line "'"))
        
        (if (setq parts (robust-parse-part-line line))
          (progn
            (setq parts-list (cons parts parts-list))
            (setq valid-parts-count (1+ valid-parts-count))
            (princ (strcat " -> : " (itoa (car parts)) "x" (itoa (cadr parts)) "-" (itoa (caddr parts))))
          )
          (princ " -> :  ")
        )
      )
    )
  )
  
  (princ (strcat "\n  : " (itoa valid-parts-count)))
  
  (if (> valid-parts-count 0)
    (reverse parts-list)
    nil
  )
)


(defun robust-parse-part-line (line / x-pos dash-pos width-str height-str quantity-str width height quantity)

  (setq line (vl-string-translate " " "" line))
  
  (setq x-pos (vl-string-position (ascii "x") line))
  (if (not x-pos) (setq x-pos (vl-string-position (ascii "*") line)))
  
  (setq dash-pos (vl-string-position (ascii "-") line))
  
  (if (and x-pos dash-pos (> x-pos 0) (> dash-pos x-pos))
    (progn
      (setq width-str (substr line 1 x-pos))
      (setq height-str (substr line (+ x-pos 2) (- dash-pos x-pos 1)))
      (setq quantity-str (substr line (+ dash-pos 2)))
      

      (if (and width-str height-str quantity-str
               (is-positive-integer width-str)
               (is-positive-integer height-str) 
               (is-positive-integer quantity-str))
        (progn
          (setq width (atoi width-str))
          (setq height (atoi height-str))
          (setq quantity (atoi quantity-str))
          (list width height quantity)
        )
        nil
      )
    )
    nil
  )
)


(defun is-positive-integer (str / i char code valid)
  (setq valid T)
  (setq i 1)
  
  (while (and valid (<= i (strlen str)))
    (setq char (substr str i 1))
    (setq code (ascii char))
    
    (if (or (< code 48) (> code 57))  
      (setq valid nil)
    )
    
    (setq i (1+ i))
  )
  
  (and valid (> (strlen str) 0))
)


(defun parse-manual-input (input-text / lines parts-list line parts valid-parts-count)
  (princ "\n   ...")
  (setq lines (split-string input-text "\n"))
  (setq parts-list '())
  (setq valid-parts-count 0)
  
  (princ (strcat "\n : " (itoa (length lines))))
  
  (foreach line lines
    (setq line (vl-string-trim " \t\r" line))
    (princ (strcat "\n : '" line "'"))
    
    (if (and (> (strlen line) 0) (setq parts (parse-part-line line)))
      (progn
        (setq parts-list (cons parts parts-list))
        (setq valid-parts-count (1+ valid-parts-count))
        (princ (strcat "\n  : " (itoa (car parts)) "x" (itoa (cadr parts)) "-" (itoa (caddr parts))))
      )
      (if (> (strlen line) 0)
        (princ (strcat "\n   : '" line "'"))
      )
    )
  )
  
  (princ (strcat "\n  : " (itoa valid-parts-count)))
  
  (if (> valid-parts-count 0)
    (reverse parts-list)
    nil
  )
)


(defun parse-part-line (line / x-pos dash-pos width-str height-str quantity-str width height quantity)
  (setq x-pos (vl-string-position (ascii "x") line))
  (if (not x-pos) (setq x-pos (vl-string-position (ascii "*") line)))
  
  (setq dash-pos (vl-string-position (ascii "-") line))
  
  (if (and x-pos dash-pos (> x-pos 0) (> dash-pos x-pos))
    (progn
      (setq width-str (substr line 1 x-pos))
      (setq height-str (substr line (+ x-pos 2) (- dash-pos x-pos 1)))
      (setq quantity-str (substr line (+ dash-pos 2)))
      
      (if (and width-str height-str quantity-str
               (numberp (setq width (atoi width-str)))
               (numberp (setq height (atoi height-str)))
               (numberp (setq quantity (atoi quantity-str)))
               (> width 0) (> height 0) (> quantity 0))
        (list width height quantity)
        nil
      )
    )
    nil
  )
)


(defun manual-parts-to-string (parts-list / result first)
  (setq result "")
  (setq first T)
  
  (foreach part parts-list
    (if first
      (setq result (strcat (itoa (car part)) "x" (itoa (cadr part)) "-" (itoa (caddr part)))
            first nil)
      (setq result (strcat result "," (itoa (car part)) "x" (itoa (cadr part)) "-" (itoa (caddr part))))
    )
  )
  
  result
)


(defun split-string (str delimiter / pos result)
  (setq result '())
  (while (setq pos (vl-string-search delimiter str))
    (setq result (cons (substr str 1 pos) result))
    (setq str (substr str (+ pos 1 (strlen delimiter))))
  )
  (reverse (cons str result))
)


(defun calculate_nesting_action ()
  (if (and (null *selected-objects*) (null *manual-parts*))
    (progn
      (alert "      !")
      (c:CUTTING)
    )
    (progn

      (update_objects_count)
      
      (if (save-settings)
        (princ "\n?    .")
        (princ "\n?      .")
      )
      (calculate_nesting)
    )
  )
)


(defun update_objects_count ()
  (if *manual-parts*

    (progn
      (setq total_manual_count 0)
      (foreach part *manual-parts*
        (setq total_manual_count (+ total_manual_count (caddr part)))
      )
      (princ (strcat "\n : " (itoa total_manual_count) " "))
    )

    (princ (strcat "\n : " (itoa (length *selected-objects*))))
  )
)


(defun ssget-to-list (ss / i lst)
  (setq i 0)
  (repeat (sslength ss)
    (setq lst (cons (ssname ss i) lst))
    (setq i (1+ i))
  )
  (reverse lst)
)


(defun calculate_nesting (/ parts-data request-data response insertion-point)
  
  (if (or (= *material-width* 0) (= *material-height* 0))
    (progn
      (alert "!    .")
      (c:CUTTING)
      (exit)
    )
  )
  
  (if (= *api-key* "")
    (progn
      (alert "!  API .")
      (c:CUTTING)
      (exit)
    )
  )
  
  (setq insertion-point (getpoint "\n    : "))
  (if (not insertion-point)
    (progn
      (princ "\n .")
      (exit)
    )
  )
  
  (princ "\n\n  ...")
  
  (if *manual-parts*
    (progn
      (princ "\n   ")
      (setq parts-data (create-parts-from-manual *manual-parts*))
    )
    (progn
      (princ "\n  ")
      (setq parts-data (extract-parts-data *selected-objects*))
    )
  )
  
  (if (not parts-data)
    (progn
      (alert "    !")
      (c:CUTTING)
      (exit)
    )
  )
  
  (princ "\n   API...")
  
  (setq request-data (create-api-request parts-data))
  
  (princ "\n   API...")
  
  (setq response (send-api-request request-data))
  
  (if response
    (progn
      (princ "\n ...")
      (process-api-response response insertion-point)
    )
    (progn
      (alert ": API !      API .")
      (c:CUTTING)
    )
  )
)


(defun create-parts-from-manual (manual-parts / parts-data width height quantity)
  (setq parts-data '())
  
  (foreach part manual-parts
    (setq width (car part))
    (setq height (cadr part))
    (setq quantity (caddr part))
    
    (setq parts-data 
      (cons 
        (list
          (cons 'width width)
          (cons 'height height)
          (cons 'quantity quantity)
          (cons 'id (strcat (itoa width) "x" (itoa height)))
          (cons 'key (strcat (itoa width) "x" (itoa height)))
        )
        parts-data
      )
    )
  )
  
  (princ (strcat "\n    : " (itoa (length parts-data)) "  "))
  parts-data
)


(defun extract-parts-data (objects-list / parts-data obj ent_data vertices bbox width height poly-area perimeter semi-p disc s1 s2 calc-diag real-diag)
  (setq parts-data '())
  
  (foreach obj objects-list
    (setq ent_data (entget obj))
    
    (if (= (cdr (assoc 0 ent_data)) "LWPOLYLINE")
      (progn
        (setq vertices (get-vertices ent_data))
        

        (if (and (> (length vertices) 2)
                 (= (cdr (assoc 70 ent_data)) 1))
          (progn

            (setq bbox (get-poly-bbox vertices))
            (setq width (- (cadr bbox) (car bbox)))  
            (setq height (- (cadddr bbox) (caddr bbox))) 
            (setq poly-area (get-poly-area vertices)) 
            

            (if (< (abs (- (* width height) poly-area)) 1.0)
              (progn


                (setq parts-data 
                  (cons 
                    (list
                      (cons 'width (fix width))
                      (cons 'height (fix height))
                      (cons 'quantity 1)
                      (cons 'id (cdr (assoc 5 ent_data)))
                      (cons 'rotated nil) 
                    )
                    parts-data
                  )
                )
              )
              (progn


                (setq perimeter (get-poly-perimeter vertices))
                (setq semi-p (/ perimeter 2.0))
                


                (setq disc (- (* semi-p semi-p) (* 4.0 poly-area)))
                
                (if (>= disc 0)
                  (progn

                    (setq s1 (/ (- semi-p (sqrt disc)) 2.0))
                    (setq s2 (/ (+ semi-p (sqrt disc)) 2.0))
                    


                    (setq calc-diag (sqrt (+ (* s1 s1) (* s2 s2))))
                    (setq real-diag (get-max-distance vertices))
                    
                    (if (< (abs (- calc-diag real-diag)) 2.0) 
                      (progn


                        (if (< s1 s2)
                          (progn (setq width s1) (setq height s2))
                          (progn (setq width s2) (setq height s1))
                        )
                        
                        (setq parts-data 
                          (cons 
                            (list
                              (cons 'width (fix width))
                              (cons 'height (fix height))
                              (cons 'quantity 1)
                              (cons 'id (cdr (assoc 5 ent_data)))
                              (cons 'rotated T) 
                            )
                            parts-data
                          )
                        )
                      )
                      (princ (strcat "\n:  " (cdr (assoc 5 ent_data)) "    (  )"))
                    )
                  )
                  (princ (strcat "\n:  " (cdr (assoc 5 ent_data)) "    "))
                )
              )
            )
          )
          (princ (strcat "\n:  " (cdr (assoc 5 ent_data)) "     "))
        )
      )
    )
  )
  
  (setq parts-data (group-similar-parts parts-data))
  parts-data
)


(defun get-poly-bbox (vertices / min-x max-x min-y max-y x y)
  (setq min-x (car (car vertices)))
  (setq max-x min-x)
  (setq min-y (cadr (car vertices)))
  (setq max-y min-y)
  
  (foreach v (cdr vertices)
    (setq x (car v))
    (setq y (cadr v))
    (if (< x min-x) (setq min-x x))
    (if (> x max-x) (setq max-x x))
    (if (< y min-y) (setq min-y y))
    (if (> y max-y) (setq max-y y))
  )
  (list min-x max-x min-y max-y)
)


(defun get-poly-area (vertices / area i p1 p2 n)
  (setq area 0.0)
  (setq n (length vertices))
  (setq i 0)
  (while (< i n)
    (setq p1 (nth i vertices))
    (setq p2 (nth (rem (1+ i) n) vertices))
    (setq area (+ area (- (* (car p1) (cadr p2)) (* (car p2) (cadr p1)))))
    (setq i (1+ i))
  )
  (abs (/ area 2.0))
)


(defun get-poly-perimeter (vertices / perim i p1 p2 n)
  (setq perim 0.0)
  (setq n (length vertices))
  (setq i 0)
  (while (< i n)
    (setq p1 (nth i vertices))
    (setq p2 (nth (rem (1+ i) n) vertices))
    (setq perim (+ perim (distance-2d p1 p2)))
    (setq i (1+ i))
  )
  perim
)


(defun distance-2d (p1 p2)
  (sqrt (+ (expt (- (car p1) (car p2)) 2) 
           (expt (- (cadr p1) (cadr p2)) 2)))
)


(defun get-max-distance (vertices / max-d d p1 p2 i j n)
  (setq max-d 0.0)
  (setq n (length vertices))
  (setq i 0)
  

  (while (< i n)
    (setq p1 (nth i vertices))
    (setq j (+ i 1))
    (while (< j n)
      (setq p2 (nth j vertices))
      (setq d (distance-2d p1 p2))
      (if (> d max-d) (setq max-d d))
      (setq j (1+ j))
    )
    (setq i (1+ i))
  )
  max-d
)


(defun get-vertices (ent_data / vertices)
  (setq vertices '())
  (foreach code ent_data
    (if (= (car code) 10)
      (setq vertices (cons (cdr code) vertices))
    )
  )
  (reverse vertices)
)


(defun group-similar-parts (parts-data / grouped-parts part width height key temp-list item existing count)
  (setq temp-list '())
  
  (foreach part parts-data
    (setq width (cdr (assoc 'width part)))
    (setq height (cdr (assoc 'height part)))
    (setq key (strcat (itoa width) "x" (itoa height)))
    
    (if (setq existing (assoc key temp-list))
      (setq temp-list (subst (cons key (1+ (cdr existing))) existing temp-list))
      (setq temp-list (cons (cons key 1) temp-list))
    )
  )
  
  (setq grouped-parts '())
  (foreach item temp-list
    (setq key (car item))
    (setq count (cdr item))
    (setq width (atoi (substr key 1 (vl-string-search "x" key))))
    (setq height (atoi (substr key (+ (vl-string-search "x" key) 2))))
    
    (setq grouped-parts 
      (cons 
        (list
          (cons 'width width)
          (cons 'height height)
          (cons 'quantity count)
          (cons 'id key)
          (cons 'key key)
        )
        grouped-parts
      )
    )
  )
  
  (princ (strcat "\n : " (itoa (length temp-list)) "  "))
  grouped-parts
)


(defun create-api-request (parts-data / request parts-list first-part)
  (setq parts-list '())
  
  (foreach part parts-data
    (setq width (cdr (assoc 'width part)))
    (setq height (cdr (assoc 'height part)))
    (setq quantity (cdr (assoc 'quantity part)))
    
    (setq parts-list (cons (strcat (itoa width) "x" (itoa height) "-" (itoa quantity)) parts-list))
  )
  
  (setq request
    (strcat
      "{"
      "\"sheetWidth\":" (itoa (fix *material-width*)) ","
      "\"sheetHeight\":" (itoa (fix *material-height*)) ","
      "\"trimming\":" (rtos *cutting-margin* 2 0) ","
      "\"kerf\":" (rtos *blade-thickness* 2 0) ","
      "\"cutType\":" (itoa (1+ (atoi *algorithm*))) ","
      "\"allowRotation\":" (if (= *allow-rotation* "1") "true" "false") ","
      "\"parts\":[" 
    )
  )
  
  (setq first-part t)
  (foreach part parts-list
    (if first-part
      (setq request (strcat request "\"" part "\"") first-part nil)
      (setq request (strcat request ",\"" part "\""))
    )
  )
  
  (setq request (strcat request "]}"))
  (princ (strcat "\n : " (itoa (length parts-list)) " "))
  request
)


(defun http-post-simple (url data api-key / xmlhttp response status objects_count)
  (vl-load-com)
  
  (princ "\n HTTP   API...")
  

  (setq objects_count 0)
  (if *manual-parts*

    (foreach part *manual-parts*
      (setq objects_count (+ objects_count (caddr part)))
    )

    (setq objects_count (length *selected-objects*))
  )
  
  (princ (strcat "\n  : " (itoa objects_count)))
  
  (if (setq xmlhttp (vlax-create-object "WinHttp.WinHttpRequest.5.1"))
    (progn
      (princ "\n? WinHttp  ")
      
      (vlax-invoke xmlhttp 'Open "POST" url :vlax-false)
      (princ "\n?  ")
      
      (vlax-invoke xmlhttp 'SetRequestHeader "Content-Type" "application/json")
      (vlax-invoke xmlhttp 'SetRequestHeader "x-api-key" api-key)
      (vlax-invoke xmlhttp 'SetRequestHeader "Accept" "application/json")
      

      (vlax-invoke xmlhttp 'SetRequestHeader "X-CAD-Platform" *cad-platform*)
      (vlax-invoke xmlhttp 'SetRequestHeader "X-CAD-Version" (get-cad-version))
      (vlax-invoke xmlhttp 'SetRequestHeader "X-OS-Type" (get-os-type))
      (vlax-invoke xmlhttp 'SetRequestHeader "X-Plugin-Version" (get-plugin-version))
      (vlax-invoke xmlhttp 'SetRequestHeader "X-Timestamp" (get-timestamp))
      (vlax-invoke xmlhttp 'SetRequestHeader "X-Session-ID" (get-session-id))
      (vlax-invoke xmlhttp 'SetRequestHeader "X-DWG-Hash" (get-dwg-hash))
      

      (vlax-invoke xmlhttp 'SetRequestHeader "X-Objects-Count" (itoa objects_count))
      
      (princ "\n?   ")
      
      (princ "\n ...")
      (princ (strcat "\n : " (itoa (strlen data)) " "))
      (vlax-invoke xmlhttp 'Send data)
      (princ "\n?  ")
      
      (command "_.DELAY" "500")
      
      (setq status (vlax-get xmlhttp 'Status))
      (setq response (vlax-get xmlhttp 'ResponseText))
      
      (princ (strcat "\n? HTTP : " (itoa status)))
      (princ (strcat "\n?  , : " (itoa (strlen response)) " "))
      
      (vlax-release-object xmlhttp)
      
      response
    )
    (progn
      (princ "\n?    WinHttp ")
      nil
    )
  )
)


(defun send-api-request (request-data / http-response)
  (princ "\n===    API ===")
  (princ (strcat "\nURL: " *api-url*))
  (princ "\n :")
  (princ request-data)
  
  (setq http-response (http-post-simple *api-url* request-data *api-key*))
  
  (if http-response
    (progn
      (princ "\n  API  !")
      http-response
    )
    (progn
      (princ "\n    API!")
      nil
    )
  )
)


(defun check-api-error (response / msg-start msg-end found-msg)
  (princ "\n   ...")
  
  (if (or (vl-string-search "\"status\":\"error\"" response)
          (vl-string-search "\"status\": \"error\"" response))
    (progn
      (princ "\n?     API")
      
      (setq msg-start (vl-string-search "\"message\": \"" response))
      (if (not msg-start)
        (setq msg-start (vl-string-search "\"message\":\"" response))
      )
      
      (if msg-start
        (progn
          (if (vl-string-search "\"message\": \"" response)
            (setq msg-start (+ msg-start 12))
            (setq msg-start (+ msg-start 11))
          )
          
          (setq msg-end (vl-string-search "\"" response msg-start))
          
          (if msg-end
            (progn
              (setq found-msg (substr response (1+ msg-start) (- msg-end msg-start)))
              (princ (strcat "\n?    : " found-msg))
              found-msg
            )
            "   .  ."
          )
        )
        "   .  ."
      )
    )
    nil
  )
)


(defun process-api-response (response insertion-point / json-data error-msg)
  (princ "\n  API...")
  
  (if (setq error-msg (check-api-error response))
    (progn
      (princ "\n?  ")
      (alert error-msg)
      (c:CUTTING)
    )
    (progn
      (setq json-data (try-parse-json response))
      
      (if json-data
        (progn
          (princ "\n? JSON  ")
          (draw-from-json json-data insertion-point)
          (princ "\n?   !")
          (alert " !\n    .")
        )
        (progn
          (princ "\n?     API")
          (alert ":      API!    .")
          (c:CUTTING)
        )
      )
    )
  )
)


(defun try-parse-json (response / sheets metrics)
  (princ "\n  JSON   API...")
  
  (if (vl-string-search "success" response)
    (progn
      (princ "\n?  API: success")
      
      (setq sheets (simple-extract-sheets response))
      (princ (strcat "\n?  : " (itoa (length sheets))))
      
      (setq metrics (extract-metrics-simple response))
      
      (if metrics
        (progn
          (princ "\n?    (GLOBAL)")
          (princ (strcat "\n?  : " (itoa (length metrics))))
        )
        (progn
          (princ "\n?   ")
        )
      )
      
      (if sheets
        (list 
          (cons 'sheets sheets)
          (cons 'metrics metrics)
        )
        nil
      )
    )
    (progn
      (princ "\n? API   ")
      nil
    )
  )
)


(defun extract-metrics-simple (response / metrics totalSheets-pos metrics-pos brace-pos end-pos metrics-content pos last-metrics-pos)
  (setq metrics '())
  (princ "\n===   ===")
  
  (princ "\n     API...")
  

  (setq totalSheets-pos (vl-string-search "\"totalSheets\":" response))
  
  (if totalSheets-pos
    (progn

       (setq pos 0)
       (setq last-metrics-pos nil)
       (while (setq pos (vl-string-search "\"metrics\":" response pos))
         (if (< pos totalSheets-pos)
           (setq last-metrics-pos pos)
         )
         (setq pos (+ pos 9))
       )
       
       (if last-metrics-pos
         (progn
           (setq brace-pos (vl-string-search "{" response last-metrics-pos))
           (if brace-pos
             (progn
               (setq end-pos (find-object-end response brace-pos))
               (if end-pos
                 (setq metrics-content (substr response (1+ brace-pos) (- end-pos brace-pos)))
               )
             )
           )
         )
         (princ "\n?    metrics  totalSheets")
       )
    )
    (princ "\n? totalSheets   (   API)")
  )
  
 

  (if (null metrics-content)
     (setq metrics-content response)
  )


  (setq totalSheets (extract-number-after metrics-content 0 "\"totalSheets\":"))
  (if totalSheets
    (progn
      (setq metrics (cons (cons 'totalSheets totalSheets) metrics))
      (princ (strcat "\n? totalSheets: " (itoa totalSheets)))
    )
  )
  
  (setq materialUsage (extract-float-after metrics-content 0 "\"materialUsage\":"))
  (if materialUsage
    (progn
      (setq metrics (cons (cons 'materialUsage materialUsage) metrics))
      (princ (strcat "\n? materialUsage: " (rtos materialUsage 2 2)))
    )
  )
  
  (setq totalParts (extract-number-after metrics-content 0 "\"totalParts\":"))
  (if totalParts
    (progn
      (setq metrics (cons (cons 'totalParts totalParts) metrics))
      (princ (strcat "\n? totalParts: " (itoa totalParts)))
    )
  )
  
  (setq wasteArea (extract-number-after metrics-content 0 "\"wasteArea\":"))
  (if wasteArea
    (progn
      (setq metrics (cons (cons 'wasteArea wasteArea) metrics))
      (princ (strcat "\n? wasteArea: " (itoa wasteArea)))
    )
  )
  
  (setq usedArea (extract-number-after metrics-content 0 "\"usedArea\":"))
  (if usedArea
    (progn
      (setq metrics (cons (cons 'usedArea usedArea) metrics))
      (princ (strcat "\n? usedArea: " (itoa usedArea)))
    )
  )
  
  (setq totalPerimeterCuts (extract-number-after metrics-content 0 "\"totalPerimeterCuts\":"))
  (if totalPerimeterCuts
    (progn
      (setq metrics (cons (cons 'totalPerimeterCuts totalPerimeterCuts) metrics))
      (princ (strcat "\n? totalPerimeterCuts: " (itoa totalPerimeterCuts)))
    )
  )
  
  (setq totalPartCuts (extract-number-after metrics-content 0 "\"totalPartCuts\":"))
  (if totalPartCuts
    (progn
      (setq metrics (cons (cons 'totalPartCuts totalPartCuts) metrics))
      (princ (strcat "\n? totalPartCuts: " (itoa totalPartCuts)))
    )
  )
  
  (setq totalTotalCuts (extract-number-after metrics-content 0 "\"totalTotalCuts\":"))
  (if totalTotalCuts
    (progn
      (setq metrics (cons (cons 'totalTotalCuts totalTotalCuts) metrics))
      (princ (strcat "\n? totalTotalCuts: " (itoa totalTotalCuts)))
    )
  )
  
  (setq totalPartsPerimeter (extract-number-after metrics-content 0 "\"totalPartsPerimeter\":"))
  (if totalPartsPerimeter
    (progn
      (setq metrics (cons (cons 'totalPartsPerimeter totalPartsPerimeter) metrics))
      (princ (strcat "\n? totalPartsPerimeter: " (itoa totalPartsPerimeter)))
    )
  )
  
  (princ (strcat "\n===   : " (itoa (length metrics)) " ==="))
  metrics
)


(defun extract-number-after (response start-pos search-str / num-start num-end num-str char found-pos)
  (setq found-pos (vl-string-search search-str response start-pos))
  (if (not found-pos)
    (return nil)
  )
  
  (setq num-start (+ found-pos (strlen search-str)))
  
  (while (and (< num-start (strlen response))
              (setq char (substr response (1+ num-start) 1))
              (or (= char " ") (= char ":") (= char "\"") (= char ",")))
    (setq num-start (1+ num-start))
  )
  
  (setq num-end num-start)
  (while (and (< num-end (strlen response))
              (setq char (substr response (1+ num-end) 1))
              (<= 48 (ascii char) 57))
    (setq num-end (1+ num-end))
  )
  
  (setq num-str (substr response (1+ num-start) (- num-end num-start)))
  
  (if (and num-str (> (strlen num-str) 0))
    (atoi num-str)
    nil
  )
)


(defun extract-float-after (response start-pos search-str / num-start num-end num-str char found-pos)
  (setq found-pos (vl-string-search search-str response start-pos))
  (if (not found-pos)
    (return nil)
  )
  
  (setq num-start (+ found-pos (strlen search-str)))
  
  (while (and (< num-start (strlen response))
              (setq char (substr response (1+ num-start) 1))
              (or (= char " ") (= char ":") (= char "\"") (= char ",")))
    (setq num-start (1+ num-start))
  )
  
  (setq num-end num-start)
  (while (and (< num-end (strlen response))
              (setq char (substr response (1+ num-end) 1))
              (or (<= 48 (ascii char) 57) (= char ".")))
    (setq num-end (1+ num-end))
  )
  
  (setq num-str (substr response (1+ num-start) (- num-end num-start)))
  
  (if (and num-str (> (strlen num-str) 0))
    (atof num-str)
    nil
  )
)


(defun simple-extract-sheets (response / sheets sheet-data)
  (setq sheets '())
  
  (setq pos 0)
  (while (setq sheet-start (vl-string-search "\"sheetWidth\":" response pos))
    (setq sheet-data (simple-extract-sheet response sheet-start))
    (if sheet-data
      (setq sheets (cons sheet-data sheets))
    )
    (setq pos (+ sheet-start 13))
  )
  
  (reverse sheets)
)


(defun simple-extract-sheet (response start-pos / sheet-data width height parts waste)
  (setq sheet-data '())
  
  (setq width (extract-number-after response start-pos "\"sheetWidth\":"))
  (if width
    (setq sheet-data (cons (cons 'sheetWidth width) sheet-data))
  )
  
  (if (setq height-pos (vl-string-search "\"sheetHeight\":" response start-pos))
    (progn
      (setq height (extract-number-after response height-pos "\"sheetHeight\":"))
      (if height
        (setq sheet-data (cons (cons 'sheetHeight height) sheet-data))
      )
    )
  )
  
  (if (setq parts-pos (vl-string-search "\"parts\":" response start-pos))
    (progn
      (setq parts (simple-extract-parts response parts-pos))
      (if parts
        (setq sheet-data (cons (cons 'parts parts) sheet-data))
      )
    )
  )
  
  (if (setq waste-pos (vl-string-search "\"waste\":" response start-pos))
    (progn
      (setq waste (simple-extract-waste response waste-pos))
      (if waste
        (setq sheet-data (cons (cons 'waste waste) sheet-data))
      )
    )
  )
  
  sheet-data
)


(defun simple-extract-parts (response start-pos / parts parts-start bracket-pos end-pos parts-content x-pos pos part-data)
  (setq parts '())
  
  (if (setq parts-start (vl-string-search "\"parts\"" response start-pos))
    (progn
      (setq bracket-pos (vl-string-search "[" response (+ parts-start 7)))
      (if bracket-pos
        (progn
          (setq end-pos (find-array-end response bracket-pos))
          (if end-pos
            (progn
              (setq parts-content (substr response (1+ bracket-pos) (- end-pos bracket-pos)))
              (setq pos 0)
              (while (and (setq x-pos (vl-string-search "\"x\":" parts-content pos))
                          (< x-pos (strlen parts-content)))
                (setq part-data (simple-extract-part parts-content x-pos))
                (if part-data
                  (setq parts (cons part-data parts))
                )
                (setq pos (+ x-pos 4))
              )
            )
          )
        )
      )
    )
  )
  (reverse parts)
)


(defun simple-extract-waste (response start-pos / waste waste-start bracket-pos end-pos waste-content x-pos pos waste-data)
  (setq waste '())
  
  (if (setq waste-start (vl-string-search "\"waste\"" response start-pos))
    (progn
      (setq bracket-pos (vl-string-search "[" response (+ waste-start 7)))
      (if bracket-pos
        (progn
          (setq end-pos (find-array-end response bracket-pos))
          (if end-pos
            (progn
              (setq waste-content (substr response (1+ bracket-pos) (- end-pos bracket-pos)))
              (setq pos 0)
              (while (and (setq x-pos (vl-string-search "\"x\":" waste-content pos))
                          (< x-pos (strlen waste-content)))
                (setq waste-data (simple-extract-part waste-content x-pos))
                (if waste-data
                  (setq waste (cons waste-data waste))
                )
                (setq pos (+ x-pos 4))
              )
            )
          )
        )
      )
    )
  )
  (reverse waste)
)


(defun simple-extract-part (response start-pos / part-data x y w h rotated rotated-pos rotated-str)
  (setq part-data '())
  
  (setq x (extract-number-after response start-pos "\"x\":"))
  (if x
    (setq part-data (cons (cons 'x x) part-data))
  )
  
  (if (setq y-pos (vl-string-search "\"y\":" response start-pos))
    (progn
      (setq y (extract-number-after response y-pos "\"y\":"))
      (if y
        (setq part-data (cons (cons 'y y) part-data))
      )
    )
  )
  
  (if (setq w-pos (vl-string-search "\"w\":" response start-pos))
    (progn
      (setq w (extract-number-after response w-pos "\"w\":"))
      (if w
        (setq part-data (cons (cons 'w w) part-data))
      )
    )
  )
  
  (if (setq h-pos (vl-string-search "\"h\":" response start-pos))
    (progn
      (setq h (extract-number-after response h-pos "\"h\":"))
      (if h
        (setq part-data (cons (cons 'h h) part-data))
      )
    )
  )
  
  (if (setq rotated-pos (vl-string-search "\"rotated\":" response start-pos))
    (progn
      (setq rotated-str (extract-bool-after response rotated-pos "\"rotated\":"))
      (setq rotated (if (eq rotated-str "true") T nil))
      (setq part-data (cons (cons 'rotated rotated) part-data))
    )
  )
  
  part-data
)


(defun extract-bool-after (response start-pos search-str / bool-start bool-end bool-str char found-pos)
  (setq found-pos (vl-string-search search-str response start-pos))
  (if (not found-pos)
    (return nil)
  )
  
  (setq bool-start (+ found-pos (strlen search-str)))
  
  (while (and (< bool-start (strlen response))
              (setq char (substr response (1+ bool-start) 1))
              (or (= char " ") (= char ":") (= char "\"")))
    (setq bool-start (1+ bool-start))
  )
  
  (setq bool-end bool-start)
  (while (and (< bool-end (strlen response))
              (setq char (substr response (1+ bool-end) 1))
              (not (or (= char ",") (= char "}") (= char "\""))))
    (setq bool-end (1+ bool-end))
  )
  
  (setq bool-str (substr response (1+ bool-start) (- bool-end bool-start)))
  
  (if (and bool-str (> (strlen bool-str) 0))
    bool-str
    nil
  )
)


(defun find-array-end (response start-pos / bracket-count pos char)
  (setq bracket-count 1)
  (setq pos (1+ start-pos))
  
  (while (and (< pos (strlen response)) (> bracket-count 0))
    (setq char (substr response (1+ pos) 1))
    
    (cond
      ((= char "[") (setq bracket-count (1+ bracket-count)))
      ((= char "]") (setq bracket-count (1- bracket-count)))
    )
    
    (setq pos (1+ pos))
  )
  
  (if (= bracket-count 0)
    (1- pos)
    nil
  )
)


(defun find-object-end (response start-pos / bracket-count pos char)
  (setq bracket-count 1)
  (setq pos (1+ start-pos))
  
  (while (and (< pos (strlen response)) (> bracket-count 0))
    (setq char (substr response (1+ pos) 1))
    
    (cond
      ((= char "{") (setq bracket-count (1+ bracket-count)))
      ((= char "}") (setq bracket-count (1- bracket-count)))
    )
    
    (setq pos (1+ pos))
  )
  
  (if (= bracket-count 0)
    (1- pos)
    nil
  )
)


(defun draw-from-json (json-data insertion-point / sheets metrics sheet sheet-data parts waste total-parts base-point sheet-offset sheet-width sheet-index max-height table-point)
  (setq sheets (cdr (assoc 'sheets json-data)))
  (setq metrics (cdr (assoc 'metrics json-data)))
  
  (princ (strcat "\n?    : " (itoa (length sheets))))
  
  (princ "\n===    ===")
  (princ (strcat "\n show-metrics: " *show-metrics*))
  (princ (strcat "\n  : " (if metrics "" "")))
  (if metrics
    (progn
      (princ (strcat "\n : " (itoa (length metrics))))
      (princ "\n   (     )")
    )
  )
  (princ "\n==============================")
  
  (if (= (length sheets) 0)
    (progn
      (princ "\n?    !")
      (exit)
    )
  )
  
  (setq base-point insertion-point)
  (setq sheet-offset 0)
  (setq total-parts 0)
  (setq sheet-index 0)
  (setq max-height 0)
  
  (setq *previous-results* nil)
  
  (foreach sheet sheets
    (setq sheet-height (cdr (assoc 'sheetHeight sheet)))
    (if (> sheet-height max-height)
      (setq max-height sheet-height)
    )
  )
  
  (foreach sheet sheets
    (setq sheet-data sheet)
    (setq parts (cdr (assoc 'parts sheet)))
    (setq waste (cdr (assoc 'waste sheet)))
    (setq sheet-width (cdr (assoc 'sheetWidth sheet-data)))
    
    (princ (strcat "\n\n===  " (itoa (1+ sheet-index)) " ==="))
    (princ (strcat "\n: " (itoa sheet-width) "x" (itoa (cdr (assoc 'sheetHeight sheet-data))) " "))
    (princ (strcat "\n  : " (itoa (length parts))))
    (princ (strcat "\n  : " (itoa (length waste))))
    
    (draw-nesting-sheet sheet-data base-point sheet-offset sheet-index)
    
    (if parts
      (draw-parts parts base-point sheet-offset)
      (princ "\n?    !")
    )
    
    (if (and waste (= *show-waste* "1"))
      (draw-waste waste base-point sheet-offset)
      (princ "\n?   ()")
    )
    
    (setq total-parts (+ total-parts (length parts)))
    
    (setq sheet-offset (+ sheet-offset sheet-width 100))
    (setq sheet-index (1+ sheet-index))
    
    (setq *previous-results* (cons sheet-data *previous-results*))
  )
  
  (princ (strcat "\n\n?   : " (itoa total-parts)))
  
  (princ (strcat "\n    : " (if metrics " " " ")))

  (if (and metrics (= *show-metrics* "1"))
    (progn
      (princ "\n   ...")
      (setq table-point (list (+ (car base-point) sheet-offset) (+ (cadr base-point) 275)))
      (draw-metrics-table-simple metrics table-point)
      (princ "\n?    ")
    )
    (progn
      (princ "\n    :")
      (cond
        ((not metrics) (princ " -     "))
        ((not (= *show-metrics* "1")) (princ " -  show-metrics "))
        (t (princ " -  "))
      )
    )
  )
)


(defun draw-nesting-sheet (sheet-data base-point offset sheet-index / width height sheet-point text-point)
  (setq width (cdr (assoc 'sheetWidth sheet-data)))
  (setq height (cdr (assoc 'sheetHeight sheet-data)))
  
  (setq sheet-point (list (+ (car base-point) offset) (cadr base-point)))
  
  (command "_.RECTANGLE" 
           "_non" sheet-point
           "_non" (list (+ (car sheet-point) width) (+ (cadr sheet-point) height)))
  
  (entmod (append (entget (entlast)) (list (cons 62 8))))
  
  (setq text-point (list (car sheet-point) (+ (cadr sheet-point) height 20)))
  
  (command "_.TEXT" 
           "_non" text-point
           "15" "0" (strcat (itoa width) "x" (itoa height) ",  " (itoa (1+ sheet-index))))
  
  (entmod (append (entget (entlast)) (list (cons 62 8))))
)


(defun draw-parts (parts base-point offset / part x y width height part-point part-point2 color-index colors part-num current-color rotated)
  (setq colors '(1 2 3 4 5 6))
  (setq color-index 0)
  (setq part-num 0)
  
  (foreach part parts
    (setq x (cdr (assoc 'x part)))
    (setq y (cdr (assoc 'y part)))
    (setq width (cdr (assoc 'w part)))
    (setq height (cdr (assoc 'h part)))
    (setq rotated (cdr (assoc 'rotated part)))
    (setq part-num (1+ part-num))
    
    (setq part-point (list (+ (car base-point) offset x) (+ (cadr base-point) y)))
    (setq part-point2 (list (+ (car part-point) width) (+ (cadr part-point) height)))
    
    (command "_.RECTANGLE" "_non" part-point "_non" part-point2)
    
    (if (= *one-color* "1")
      (setq current-color 1)
      (setq current-color (nth color-index colors))
    )
    
    (entmod (append (entget (entlast)) (list (cons 62 current-color))))
    
    (command "_.TEXT" 
             "_non" (list (+ (car part-point) 10) (+ (cadr part-point) 10))
             "15" "0" (strcat (itoa width) "x" (itoa height)))
    
    (if rotated
      (command "_.TEXT"
               "_non" (list (- (car part-point2) 20) (- (cadr part-point2) 20))
               "15" "0" "R")
    )
    
    (if (not (= *one-color* "1"))
      (setq color-index (rem (1+ color-index) (length colors)))
    )
  )
  
  (princ (strcat "\n   : " (itoa part-num)))
)


(defun draw-waste (waste base-point offset / waste-item x y width height waste-point waste-point2 waste-num)
  (setq waste-num 0)
  
  (foreach waste-item waste
    (setq x (cdr (assoc 'x waste-item)))
    (setq y (cdr (assoc 'y waste-item)))
    (setq width (cdr (assoc 'w waste-item)))
    (setq height (cdr (assoc 'h waste-item)))
    (setq waste-num (1+ waste-num))
    
    (setq waste-point (list (+ (car base-point) offset x) (+ (cadr base-point) y)))
    (setq waste-point2 (list (+ (car waste-point) width) (+ (cadr waste-point) height)))
    
    (command "_.RECTANGLE" "_non" waste-point "_non" waste-point2)
    (entmod (append (entget (entlast)) (list (cons 62 253))))
  )
  
  (princ (strcat " : " (itoa waste-num)))
)


(defun draw-metrics-table-simple (metrics table-point / start-x start-y current-y row-height value value-sq-m value-m)
  (setq start-x (car table-point))
  (setq start-y (cadr table-point))
  (setq current-y start-y)
  (setq row-height 30)
  
  (princ "\n===   ===")
  
  (command "_.TEXT" "_non" (list start-x current-y) "20" "0" " ")
  (setq current-y (- current-y row-height))
  
  (if metrics
    (progn
      (if (setq value (cdr (assoc 'totalSheets metrics)))
        (progn
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" (strcat " : " (itoa value)))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'materialUsage metrics)))
        (progn
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" (strcat " : " (rtos value 2 2) "%"))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'totalParts metrics)))
        (progn
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" (strcat " : " (itoa value)))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'wasteArea metrics)))
        (progn
          (setq value-sq-m (/ value 1000000.0))
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" 
                   (strcat " : " (rtos value-sq-m 2 2) " . (" (itoa value) " .)"))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'usedArea metrics)))
        (progn
          (setq value-sq-m (/ value 1000000.0))
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" 
                   (strcat " : " (rtos value-sq-m 2 2) " . (" (itoa value) " .)"))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'totalPerimeterCuts metrics)))
        (progn
          (setq value-m (/ value 1000.0))
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" 
                   (strcat "  : " (rtos value-m 2 3) "  (" (itoa value) " )"))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'totalPartCuts metrics)))
        (progn
          (setq value-m (/ value 1000.0))
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" 
                   (strcat "  : " (rtos value-m 2 3) "  (" (itoa value) " )"))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'totalTotalCuts metrics)))
        (progn
          (setq value-m (/ value 1000.0))
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" 
                   (strcat "  : " (rtos value-m 2 3) "  (" (itoa value) " )"))
          (setq current-y (- current-y row-height))
        )
      )
      
      (if (setq value (cdr (assoc 'totalPartsPerimeter metrics)))
        (progn
          (setq value-m (/ value 1000.0))
          (command "_.TEXT" "_non" (list start-x current-y) "15" "0" 
                   (strcat "  : " (rtos value-m 2 3) "  (" (itoa value) " )"))
          (setq current-y (- current-y row-height))
        )
      )
    )
    (princ "\n   ")
  )
  
  (princ "\n===   ===")
)


(defun c: ()
  (princ "\n  ...")
  (c:CUTTING)
)


(princ "\n  .  CUTTING    .")
(princ (strcat "\n : " (get-plugin-version)))
(load-settings)
(princ)
